package sim.lib.memory;

import java.awt.*;

import sim.*;
import sim.engine.*;

import sim.lib.wires.Junction;

public class ShiftRegister extends RegisterStructure
{
/* ==================================================================
	Creation Part
	================================================================= */
	private static Image ICON = GuiFileLink.getImage("sim/lib/memory/ShiftRegisterIcon.gif");
	
	public Image getIcon()
	{
		return ShiftRegister.ICON;
	}
	
	public String getBubbleHelp()
	{
		return "Shift Register";
	}
	
/* ==================================================================
	GUI part
	================================================================= */
	public ShiftRegister()
	{
		super();
		
		this.name = "Shift Register";
	}
	
	protected Wrapper getCopy()
	{
		ShiftRegister result = new ShiftRegister();
		
		result.changeDelay(this.delay);
		result.setBusSize(this.busSize);
		
		return result;
	}
	
	public void paint(Graphics g)
	{
		// draw if visible
		if(this.isVisible())
		{
			int gridGap = CentralPanel.ACTIVE_GRID.getCurrentGridGap();
			int increment = gridGap / 4;
			
			int width = gridGap * this.gridSize.width;
			int temp = 3 * gridGap;
			
			g.setColor(this.brush);
			
			g.drawRect(gridGap, gridGap + increment, width - 2 * gridGap, 17 * increment);
			
			g.drawLine(0, temp, gridGap, temp);
			g.drawLine(0, 4 * gridGap, gridGap, 4 * gridGap);
			g.drawLine(0, 5 * gridGap, gridGap, 5 * gridGap);
			
			g.drawLine(width - gridGap, 2 * gridGap, width, 2 * gridGap);
			g.drawLine(width - gridGap, temp, width, temp);
			g.drawLine(width - gridGap, 4 * gridGap, width, 4 * gridGap);
			g.drawLine(width - gridGap, 5 * gridGap, width, 5 * gridGap);
			
			temp = gridGap * (this.gridSize.width / 2) - 1;
			
			g.fillRect(temp, 0, 3, gridGap + increment);
			g.fillRect(temp, 22 * increment, 3, 6 * increment);
			
			g.drawRect(10 * increment, 4 * gridGap, width - 5 * gridGap, 6 * increment);
			
			g.setFont(new Font(Wrapper.FONT_NAME, Font.PLAIN, 3 * increment));
			
			FontMetrics fm = g.getFontMetrics(g.getFont());
			
			g.drawString("<<", gridGap + increment, 3 * gridGap + increment);
			g.drawString("iL", gridGap + increment, 4 * gridGap + increment);
			g.drawString("oL", gridGap + increment, 5 * gridGap + increment);
			
			g.drawString("ld", width - 5 * increment - fm.stringWidth("ld"), 2 * gridGap + increment);
			g.drawString(">>", width - 5 * increment - fm.stringWidth(">>"), 3 * gridGap + increment);
			g.drawString("iR", width - 5 * increment - fm.stringWidth("iR"), 4 * gridGap + increment);
			g.drawString("oR", width - 5 * increment - fm.stringWidth("oR"), 5 * gridGap + increment);
			
			g.clearRect(10 * increment + 1, 4 * gridGap + 1, width - 5 * gridGap - 2, 6 * increment - 2);
			
			g.drawString(this.value, (width - fm.stringWidth(this.value)) / 2, 5 * gridGap);
			
			Shape old = g.getClip();
			g.setClip(2 * gridGap, gridGap, width - 4 * gridGap, 2 * gridGap);
			g.setFont(new Font(Wrapper.FONT_NAME, Font.BOLD, 3 * increment));
			fm = g.getFontMetrics(g.getFont());
			g.drawString(this.name, (width - fm.stringWidth(this.name)) / 2, 10 * increment);
			g.setClip(old);
		}
	}
	
/* ==================================================================
	Maintanance Part
	================================================================= */
	private Junction RE = null;
	private Junction LE = null;
	private Junction iR = null;
	private Junction oR = null;
	private Junction iL = null;
	private Junction oL  = null;
	
	public boolean canDrop()
	{
		boolean result = Wrapper.canDropJuncion(this.gridLocation.x, this.gridLocation.y + 3, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x, this.gridLocation.y + 4, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x, this.gridLocation.y + 5, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 2, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 3, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 4, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 5, 1);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width / 2, this.gridLocation.y, this.busSize);
		result = result && Wrapper.canDropJuncion(this.gridLocation.x + this.gridSize.width / 2, this.gridLocation.y + 7, this.busSize);
		
		return result;
	}
		
	public void droped()
	{
		this.LE = Wrapper.setPinAt(this.gridLocation.x, this.gridLocation.y + 3, 1);
		this.iL = Wrapper.setPinAt(this.gridLocation.x, this.gridLocation.y + 4, 1);
		this.oL = Wrapper.setPinAt(this.gridLocation.x, this.gridLocation.y + 5, 1);
		
		this.clock = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 2, 1);
		
		this.RE = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 3, 1);
		this.iR = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 4, 1);
		this.oR = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width, this.gridLocation.y + 5, 1);
		
		this.input = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width / 2, this.gridLocation.y, this.busSize);
		this.output = Wrapper.setPinAt(this.gridLocation.x + this.gridSize.width / 2, this.gridLocation.y + 7, this.busSize);
		
		this.changeColor(Color.black);
		this.oldValue = null;
		this.oldStore = null;
	}
	
	public void selected()
	{
		this.input.removePin();
		this.output.removePin();
		this.clock.removePin();
		this.RE.removePin();
		this.LE.removePin();
		this.iL.removePin();
		this.oL.removePin();
		this.iR.removePin();
		this.oR.removePin();
		
		this.changeColor(Color.green);
	}
	
	public void checkAfterSelected()
	{
		Wrapper.checkPin(this.input);
		Wrapper.checkPin(this.output);
		Wrapper.checkPin(this.clock);
		Wrapper.checkPin(this.RE);
		Wrapper.checkPin(this.LE);
		Wrapper.checkPin(this.iL);
		Wrapper.checkPin(this.oL);
		Wrapper.checkPin(this.iR);
		Wrapper.checkPin(this.oR);
	}
	
/* ==================================================================
	Simulation part
	================================================================= */
	protected boolean pastLE;
	protected boolean pastRE;
	private double leGoesHigh;
	private double reGoesHigh;
	private boolean handleShift;
	
	public void evaluateOutput(double currentTime, Data[] currentInputs, EnginePeer peer)
	{
		double time = this.delay + currentTime;
		boolean foundUndefined = false;
		boolean update = false;
		int loop, index, hex = 0, base = 1;
		
		// check clock
		if(!currentInputs[this.busSize].isUndefined())
		{
			if((!this.pastClock) && currentInputs[this.busSize].getValue())
				this.clkGoesHigh = currentTime;
			
			this.pastClock = currentInputs[this.busSize].getValue();
		}
		else
			this.pastClock = true;
		
		// check RE
		if(!currentInputs[this.busSize + 1].isUndefined())
		{
			if((!this.pastRE) && currentInputs[this.busSize + 1].getValue())
			{
				this.reGoesHigh = currentTime;
				this.handleShift = false;
			}
			
			this.pastRE = currentInputs[this.busSize + 1].getValue();
		}
		else
			this.pastRE = true;
		
		// check LE
		if(!currentInputs[this.busSize + 3].isUndefined())
		{
			if((!this.pastLE) && currentInputs[this.busSize + 3].getValue())
			{
				this.leGoesHigh = currentTime;
				this.handleShift = false;
			}
			
			this.pastLE = currentInputs[this.busSize + 3].getValue();
		}
		else
			this.pastLE = true;
		
		
		// load data
		if(this.clkGoesHigh == currentTime)
		{
			for(loop = 0; loop < this.busSize; loop++)
			{
				if(currentInputs[loop].isUndefined())
					this.store[loop] = 2;
				else if(currentInputs[loop].getValue())
					this.store[loop] = 1;
				else
					this.store[loop] = 0;
			}
			
			this.handleShift = false;
			update = true;
		}
		
		// need to shift?
		boolean shiftRight = (this.reGoesHigh == currentTime);
		boolean shiftLeft = (this.leGoesHigh == currentTime);
		
		// shift in one direction only
		if((shiftRight ^ shiftLeft) && (!this.handleShift))
		{
			this.handleShift = true;
			
			if(shiftRight)
			{
				if(this.store[0] == 2)
					peer.setOutputPinUndefined(this.busSize, time);
				else
					peer.setOutputPinValue(this.busSize, this.store[0] == 1, time);
				
				for(loop = 0; loop < (this.busSize - 1); loop++)
					this.store[loop] = this.store[loop + 1];
				
				peer.setOutputPinUndefined(this.busSize + 1, time);
				
				if(currentInputs[this.busSize + 2].isUndefined())
					this.store[this.busSize - 1] = 2;
				else if(currentInputs[this.busSize + 2].getValue())
					this.store[this.busSize - 1] = 1;
				else
					this.store[this.busSize - 1] = 0;
			}
			else
			{
				if(this.store[this.busSize - 1] == 2)
					peer.setOutputPinUndefined(this.busSize + 1, time);
				else
					peer.setOutputPinValue(this.busSize + 1, this.store[this.busSize - 1] == 1, time);
				
				peer.setOutputPinUndefined(this.busSize, time);
				
				for(loop = this.busSize - 1; loop > 0; loop--)
					this.store[loop] = this.store[loop - 1];
				
				if(currentInputs[this.busSize + 4].isUndefined())
					this.store[0] = 2;
				else if(currentInputs[this.busSize + 4].getValue())
					this.store[0] = 1;
				else
					this.store[0] = 0;
			}
			
			update = true;
		}
		else if(update)
		{
			peer.setOutputPinUndefined(this.busSize + 1, time);
			peer.setOutputPinUndefined(this.busSize, time);
		}
		
		// drive content and display hex value
		if(update)
		{
			hex = 0;
			base = 1;
			
			for(loop = 0; loop < (this.busSize / 4 + 1); loop++)
			{
				for(index = loop * 4; (index < 4 * (loop + 1)) && (index < this.busSize); index++)
				{
					if(this.store[index] == 2)
					{
						foundUndefined = true;
						peer.setOutputPinUndefined(index, time);
					}
					else if(this.store[index] == 1)
					{
						hex = hex + base;
						peer.setOutputPinValue(index, true, time);
					}
					else
					{
						peer.setOutputPinValue(index, false, time);
					}
					
					base = 2 * base;
				}
				
				if(loop == 0)
				{
					if(foundUndefined)
						this.value = "-";
					else
						this.value = Integer.toHexString(hex);
				}
				else if(foundUndefined)
					this.value = "-" + this.value;
				else if(this.value.length() < this.valueLenght)
					this.value = Integer.toHexString(hex) + this.value;
				
				hex = 0;
				base = 1;
				foundUndefined = false;
			}
			
			this.value = this.value.toUpperCase();
			CentralPanel.ACTIVE_GRID.paintComponent(this);
		}
	}
	
	public void createEnginePeer(EnginePeerList epl)
	{
		this.pastClock = false;
		this.clkGoesHigh = -1;
		this.pastLE = false;
		this.leGoesHigh = -1;
		this.pastRE = false;
		this.reGoesHigh = -1;
		
		this.handleShift = true;
		
		int loop;
		EnginePeer ep = new EnginePeer(this.busSize + 5, this.busSize + 2, this);
		
		for(loop = 0; loop < this.busSize; loop++)
		{
			ep.setInputPin(loop, this.input.getNodes().getItemAt(loop));
			ep.setOutputPin(loop, this.output.getNodes().getItemAt(loop));
		}
		
		ep.setInputPin(this.busSize, this.clock.getNodes().getItemAt(0));
		ep.setInputPin(this.busSize + 1, this.RE.getNodes().getItemAt(0));
		ep.setInputPin(this.busSize + 2, this.iR.getNodes().getItemAt(0));
		ep.setInputPin(this.busSize + 3, this.LE.getNodes().getItemAt(0));
		ep.setInputPin(this.busSize + 4, this.iL.getNodes().getItemAt(0));
		
		ep.setOutputPin(this.busSize, this.oR.getNodes().getItemAt(0));
		ep.setOutputPin(this.busSize + 1, this.oL.getNodes().getItemAt(0));
		
		for(loop = 0; loop < this.busSize; loop++)
		{
			if(this.store[loop] == 2)
				ep.setOutputPinUndefined(loop, 0);
			else
				ep.setOutputPinValue(loop, this.store[loop] == 1, 0);
		}
		
		ep.setOutputPinUndefined(this.busSize + 1, 0);
		ep.setOutputPinUndefined(this.busSize, 0);
		
		epl.insertItem(ep);
	}
}